<?php
// Database connection
try {
    require 'db_connect.php';
    if (!$conn) throw new Exception("Database connection is null.");
} catch (Exception $e) {
    $error_message = "Failed to connect to the database: " . $e->getMessage();
    goto render_page;
}

// Now Showing logic
date_default_timezone_set("Asia/Colombo");
$currentDay = date('l');
$currentTime = date('H:i:s');

$now_stmt = $conn->prepare("
    SELECT * FROM tv_schedule 
    WHERE day = :day 
    AND time <= :time 
    ORDER BY time DESC 
    LIMIT 1
");
$now_stmt->execute([':day'=>$currentDay, ':time'=>$currentTime]);
$now_program = $now_stmt->fetch(PDO::FETCH_ASSOC);

render_page:
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Live TV Player</title>
<script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
<style>
/* Wrapper */
.video-player-wrapper {
    max-width: 1200px;
    margin: 20px auto;
    border-radius: 20px;
    overflow: hidden;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
}

/* Player */
.video-player {
    position: relative;
    width: 100%;
    padding-top: 56.25%; /* 16:9 */
    background: #000;
}
.video-player video {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    border: none;
    background: #000;
}

/* Alert bar */
.player-alert-bar {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 20px;
    font-size: 1rem;
    font-weight: 500;
    background: linear-gradient(90deg, #fdc206, #ff9800);
    color: #000;
}
.player-alert-bar .alert-label { font-weight: 700; text-transform: uppercase; }
.player-alert-bar .alert-text { flex: 1; }
.player-alert-bar .error { color: #fff; background: rgba(220, 53, 69, 0.9); padding: 6px 10px; border-radius: 6px; }

/* Responsive */
@media (max-width: 768px) {
    .player-alert-bar { font-size: 0.9rem; padding: 10px 15px; }
}
@media (max-width: 576px) {
    .player-alert-bar { flex-direction: column; align-items: flex-start; font-size: 0.85rem; }
}
</style>
</head>
<body>

<!-- Live TV Player -->
<div class="video-player-wrapper">

    <div class="video-player">
        <video id="liveVideo" autoplay muted playsinline controls></video>
    </div>

    <div class="player-alert-bar">
        <?php if (isset($error_message)): ?>
            <span class="alert-text error"><?php echo htmlspecialchars($error_message); ?></span>
        <?php elseif ($now_program): ?>
            <span class="alert-label">🔴 Live Now</span>
            <span class="alert-text">
                <?php echo htmlspecialchars($now_program['program_name']); ?> 
                (<?php echo date('h:i A', strtotime($now_program['time'])); ?>)
            </span>
        <?php else: ?>
            <span class="alert-label">📺 Live Info</span>
            <span class="alert-text">No program currently airing.</span>
        <?php endif; ?>
    </div>
</div>

<script>
// Encoded URLs (Base64)
const encodedUrls = [
    'aHR0cHM6Ly9qazNsejh4a2x3NzktaGxzLWxpdmUuNWNlbnRzY2RuLmNvbS9saXZlLzYyMjZmN2NiZTU5ZTk5YTkwYjVjZWY2Zjk0Zjk2NmZkLnNkcC9wbGF5bGlzdC5tM3U4',
    'aHR0cHM6Ly9lZGdlMS1tb2JsaXZlLnl1cHBjZG4ubmV0L2RybS9zbWlsOnN3YXJuYXdhaGluaWRybS5zbWlsL21hbmlmZXN0Lm0zdTg='
];

// Decode Base64 to URL
function decodeUrl(encoded) {
    try {
        return atob(encoded);
    } catch(e) {
        return null;
    }
}

const video = document.getElementById('liveVideo');
let currentUrlIndex = 0;

function playStream(urlIndex) {
    const url = decodeUrl(encodedUrls[urlIndex]);
    if (!url) return;

    if (Hls.isSupported()) {
        const hls = new Hls();
        hls.loadSource(url);
        hls.attachMedia(video);

        hls.on(Hls.Events.MANIFEST_PARSED, () => video.play());

        hls.on(Hls.Events.ERROR, (event, data) => {
            if (data.fatal) {
                hls.destroy();
                if (urlIndex + 1 < encodedUrls.length) playStream(urlIndex + 1);
                else alert('Live stream is unavailable.');
            }
        });
    } else if (video.canPlayType('application/vnd.apple.mpegurl')) {
        video.src = url;
        video.addEventListener('error', () => {
            if (urlIndex + 1 < encodedUrls.length) playStream(urlIndex + 1);
            else alert('Live stream is unavailable.');
        });
        video.play();
    } else {
        alert('HLS not supported in this browser.');
    }
}

// Start playback
playStream(currentUrlIndex);
</script>

</body>
</html>
